#ifndef GLSL_COLOR
#define GLSL_COLOR

/*
color.glsl
Implementation of some color conversion equations from: http://www.brucelindbloom.com/Equations.html 
HSV, HSL and HCY implementation from wikipedia: https://en.wikipedia.org/wiki/HSL_and_HSV
*/

//HCY color space. Luma is calculated with the standard RGB weights
vec3 RGB2HCY(vec3 rgb) {
	float rgbmin = min(min(rgb.r, rgb.g), rgb.b);
	float rgbmax = max(max(rgb.r, rgb.g), rgb.b);
	float C = rgbmax - rgbmin;

	if(C <= 0.0) {
		C = 0.000001;
		//return vec3(0, 0, 0); //This is the correct implementation but it creates artifacts. We just add a very low chroma
	}

	float h;
	if(rgb.r == rgbmax) {
		h = mod((rgb.g - rgb.b) / C, 6);
	}
	else if(rgb.g == rgbmax) {
		h = 2.0 + (rgb.b - rgb.r) / C;
	}
	else {
		h = 4.0 + (rgb.r - rgb.g) / C;
	}

	h *= 60.0;
	if(h < 0) h += 360.0;

	float luma = 0.30*rgb.r + 0.59*rgb.g + 0.11*rgb.b;
	return vec3(h, C, luma);

}

vec3 HCY2RGB(vec3 hsl) {
	float C = hsl.y;
	float Hp = (hsl.x - floor(hsl.x / 360.0) * 360.0) / 60.0;
	float X = C * (1.0 - abs(mod(Hp, 2.0) - 1.0));
	vec3 rgb = vec3(0);

	if(Hp < 1.0) {
		rgb = vec3(C, X, 0);
	}
	else if(Hp < 2.0) {
		rgb = vec3(X, C, 0);
	}
	else if(Hp < 3.0) {
		rgb = vec3(0, C, X);
	}
	else if(Hp < 4.0) {
		rgb = vec3(0, X, C);
	}
	else if(Hp < 5.0) {
		rgb = vec3(X, 0, C);
	}
	else if(Hp < 6.0) {
		rgb = vec3(C, 0, X);
	}

	float m = hsl.z - (0.30*rgb.r + 0.59*rgb.g + 0.11*rgb.b);
	rgb += vec3(m);

	return clamp(rgb, vec3(0), vec3(1));
}

//HSL color space
vec3 RGB2HSL(vec3 rgb) {
	float rgbmin = min(min(rgb.r, rgb.g), rgb.b);
	float rgbmax = max(max(rgb.r, rgb.g), rgb.b);
	float chroma = rgbmax - rgbmin;

	if(chroma <= 0.0) {
		chroma = 0.000001;
		//return vec3(0, 0, 0); //This is the correct implementation but it creates artifacts. We just add a very low chroma
	}

	float h;
	if(rgb.r == rgbmax) {
		h = mod((rgb.g - rgb.b) / chroma, 6);
	}
	else if(rgb.g == rgbmax) {
		h = 2.0 + (rgb.b - rgb.r) / chroma;
	}
	else {
		h = 4.0 + (rgb.r - rgb.g) / chroma;
	}

	h *= 60.0;
	if(h < 0) h += 360.0;

	float L = (rgbmax + rgbmin) * 0.5;
	return vec3(h, chroma / (1-abs(2*L-1)), L);

}

vec3 HSL2RGB(vec3 hsl) {
	float C = (1.0 - abs(2.0*hsl.z - 1.0)) * hsl.y;
	float Hp = (hsl.x - floor(hsl.x / 360.0) * 360.0) / 60.0;
	float X = C * (1.0 - abs(mod(Hp, 2.0) - 1.0));
	vec3 rgb = vec3(0);

	if(Hp < 1.0) {
		rgb = vec3(C, X, 0);
	}
	else if(Hp < 2.0) {
		rgb = vec3(X, C, 0);
	}
	else if(Hp < 3.0) {
		rgb = vec3(0, C, X);
	}
	else if(Hp < 4.0) {
		rgb = vec3(0, X, C);
	}
	else if(Hp < 5.0) {
		rgb = vec3(X, 0, C);
	}
	else if(Hp < 6.0) {
		rgb = vec3(C, 0, X);
	}

	float m = hsl.z - 0.5*C;
	rgb += vec3(m);

	return clamp(rgb, vec3(0), vec3(1));
}

//HSY color space. Luminance is calculated with the standard RGB weights
vec3 RGB2HSY(vec3 rgb) {
	float rgbmin = min(min(rgb.r, rgb.g), rgb.b);
	float rgbmax = max(max(rgb.r, rgb.g), rgb.b);
	float chroma = rgbmax - rgbmin;

	if(chroma <= 0.0) {
		chroma = 0.000001;
		//return vec3(0, 0, 0); //This is the correct implementation but it creates artifacts. We just add a very low chroma
	}

	float h;
	if(rgb.r == rgbmax) {
		h = mod((rgb.g - rgb.b) / chroma, 6);
	}
	else if(rgb.g == rgbmax) {
		h = 2.0 + (rgb.b - rgb.r) / chroma;
	}
	else {
		h = 4.0 + (rgb.r - rgb.g) / chroma;
	}

	h *= 60.0;
	if(h < 0) h += 360.0;

	float L = 0.30*rgb.r + 0.59*rgb.g + 0.11*rgb.b;
	return vec3(h, chroma / (1 - abs(2 * L - 1)), L);
}

vec3 HSY2RGB(vec3 hsl) {
	float C = (1.0 - abs(2.0*hsl.z - 1.0)) * hsl.y;
	float Hp = (hsl.x - floor(hsl.x / 360.0) * 360.0) / 60.0;
	float X = C * (1.0 - abs(mod(Hp, 2.0) - 1.0));
	vec3 rgb = vec3(0);

	if(Hp < 1.0) {
		rgb = vec3(C, X, 0);
	}
	else if(Hp < 2.0) {
		rgb = vec3(X, C, 0);
	}
	else if(Hp < 3.0) {
		rgb = vec3(0, C, X);
	}
	else if(Hp < 4.0) {
		rgb = vec3(0, X, C);
	}
	else if(Hp < 5.0) {
		rgb = vec3(X, 0, C);
	}
	else if(Hp < 6.0) {
		rgb = vec3(C, 0, X);
	}

	float m = hsl.z - (0.30*rgb.r + 0.59*rgb.g + 0.11*rgb.b);
	rgb += vec3(m);

	return clamp(rgb, vec3(0), vec3(1));
}

//HSV color space. Luma is calculated as the max RGB value
vec3 RGB2HSV(vec3 rgb) {
	float rgbmin = min(min(rgb.r, rgb.g), rgb.b);
	float rgbmax = max(max(rgb.r, rgb.g), rgb.b);
	float chroma = rgbmax - rgbmin;

	if(chroma <= 0.0) {
		chroma = 0.000001;
		//return vec3(0, 0, 0); //This is the correct implementation but it creates artifacts. We just add a very low chroma
	}

	float h;
	if(rgb.r == rgbmax) {
		h = mod((rgb.g - rgb.b) / chroma, 6);
	}
	else if(rgb.g == rgbmax) {
		h = 2.0 + (rgb.b - rgb.r) / chroma;
	}
	else {
		h = 4.0 + (rgb.r - rgb.g) / chroma;
	}

	h *= 60.0;
	if(h < 0) h += 360.0;

	return vec3(h, chroma / rgbmax, rgbmax);

}

vec3 HSV2RGB(vec3 hsv) {
	float C = hsv.y * hsv.z;
	float Hp = (hsv.x - floor(hsv.x / 360.0) * 360.0) / 60.0;
	float X = C * (1.0 - abs(mod(Hp,2.0) - 1.0));
	vec3 rgb = vec3(0);

	if(Hp < 1.0) {
		rgb = vec3(C, X, 0);
	}
	else if(Hp < 2.0) {
		rgb = vec3(X, C, 0);
	}
	else if(Hp < 3.0) {
		rgb = vec3(0, C, X);
	}
	else if(Hp < 4.0) {
		rgb = vec3(0, X, C);
	}
	else if(Hp < 5.0) {
		rgb = vec3(X, 0, C);
	}
	else if(Hp < 6.0) {
		rgb = vec3(C, 0, X);
	}

	float m = hsv.z - C;
	rgb += vec3(m);

	return clamp(rgb, vec3(0), vec3(1));
}

vec3 XYZ2xyY(vec3 XYZ) {
	return vec3(XYZ.x / (XYZ.x + XYZ.y + XYZ.z),
				XYZ.y / (XYZ.x + XYZ.y + XYZ.z),
				XYZ.y);
}

vec3 xyY2XYZ(vec3 xyY) {
	return vec3((xyY.z / xyY.y) * xyY.x,
				xyY.z,
				(xyY.z / xyY.y) * (1.0 - xyY.x - xyY.y));
}

vec3 XYZ2Luv(vec3 XYZ, vec3 white) {
	const float eps = 0.00856;
	const float k = 903.3;
	const vec3 base = vec3(1.0,15.0,3.0);

	float yr = XYZ.y / white.y;
	float L = yr > eps ? 116*pow(yr,1.0/3.0) -16 : k*yr;
	float xyzDen = dot(XYZ,base);
	float whiteDen = dot(white,base);

	return vec3(
		L,
		(4*XYZ.x / xyzDen) - (4*white.x / whiteDen),
		(9*XYZ.y / xyzDen) - (9*white.y / whiteDen)
	);
}

vec3 Luv2XYZ(vec3 Luv, vec3 white) {
	const float eps = 0.00856;
	const float k = 903.3;
	const float ke = k * eps;
	const vec3 base = vec3(1.0,15.0,3.0);

	float whiteDen = dot(white,base);
	float Y = Luv.x > ke ? pow((Luv.x+16)/116,3) : Luv.x / k;
	float up = 4*white.x / whiteDen;
	float vp = 9*white.y / whiteDen;

	float a = (52*Luv.x /(Luv.y+13*Luv.x*up) - 1)/3.0;
	float b = -5*Y;
	const float c = -1.0/3.0;
	float d = Y * (39*Luv.x/(Luv.z + 13*Luv.x*vp) - 5);

	float X = (d - b) / (a - c);

	return vec3(
		X,
		Y,
		X*a + b
	);
}

vec3 RGB2XYZ(vec3 rgb) {
	const mat3 rgb2xyzMatrix = transpose(
			mat3(0.4124, 0.3576, 0.1805,
			0.2126, 0.7152, 0.0722,
			0.0193, 0.1192, 0.9505));
	return rgb2xyzMatrix * rgb;
}

vec3 XYZ2RGB(vec3 xyz) {
	const mat3 xyz2rgbMatrix = transpose(
			mat3(3.2406, -1.5372, -0.4986,
			-0.9689, 1.8758, 0.0415, 
			0.0557, -0.2040, 1.0570));
	return xyz2rgbMatrix * xyz;
}

vec3 RGB2xyY(vec3 rgb) {
	//Convert to XYZ first then to xyY from there
	return XYZ2xyY(RGB2XYZ(rgb));
}

vec3 xyY2RGB (vec3 xyY) {
	//Convert to XYZ first then to RGB from there
	return XYZ2RGB(xyY2XYZ(xyY));
}

vec3 RGB2sRGB(vec3 rgb) {
	const float a = 0.055;
	return vec3(
		rgb.r <= 0.0031308 ? (12.92 * rgb.r) : ((1 + a) * pow(rgb.r, 1 / 2.4) - a),
		rgb.g <= 0.0031308 ? (12.92 * rgb.g) : ((1 + a) * pow(rgb.g, 1 / 2.4) - a),
		rgb.b <= 0.0031308 ? (12.92 * rgb.b) : ((1 + a) * pow(rgb.b, 1 / 2.4) - a));
}

vec4 RGB2sRGB(vec4 rgba) {
	const float a = 0.055;
	return vec4(
		rgba.r <= 0.0031308 ? (12.92 * rgba.r) : ((1 + a) * pow(rgba.r, 1 / 2.4) - a),
		rgba.g <= 0.0031308 ? (12.92 * rgba.g) : ((1 + a) * pow(rgba.g, 1 / 2.4) - a),
		rgba.b <= 0.0031308 ? (12.92 * rgba.b) : ((1 + a) * pow(rgba.b, 1 / 2.4) - a),
		rgba.a <= 0.0031308 ? (12.92 * rgba.a) : ((1 + a) * pow(rgba.a, 1 / 2.4) - a));
}

vec3 sRGB2RGB(vec3 srgb) {
	const float a = 0.055;
	return vec3(
		srgb.r <= 0.04045 ? (srgb.r / 12.92) : (pow((srgb.r + a) / (1 + a), 2.4)),
		srgb.g <= 0.04045 ? (srgb.g / 12.92) : (pow((srgb.g + a) / (1 + a), 2.4)),
		srgb.b <= 0.04045 ? (srgb.b / 12.92) : (pow((srgb.b + a) / (1 + a), 2.4)));
}

vec4 sRGB2RGB(vec4 srgba) {
	const float a = 0.055;
	return vec4(
		srgba.r <= 0.04045 ? (srgba.r / 12.92) : (pow((srgba.r + a) / (1 + a), 2.4)),
		srgba.g <= 0.04045 ? (srgba.g / 12.92) : (pow((srgba.g + a) / (1 + a), 2.4)),
		srgba.b <= 0.04045 ? (srgba.b / 12.92) : (pow((srgba.b + a) / (1 + a), 2.4)),
		srgba.a <= 0.04045 ? (srgba.a / 12.92) : (pow((srgba.a + a) / (1 + a), 2.4)));
}

float RGB2Luminance(vec3 rgb) {
	return dot(rgb,  vec3(0.2126, 0.7152, 0.0722));
}

float RGB2LogLuminance(vec3 rgb) {
	return log(max(0.0001,dot(rgb,  vec3(0.2126, 0.7152, 0.0722))));
}

/*
color.glsl end
*/

#endif